module T_grid3D
	
	implicit none

integer, PARAMETER :: fastFloorOffset = 10000    

type grid3D                                         ! 3D grid maps of observables
  real,    dimension (3) :: Rmin, Rmax, Rspan, step ! grid size and spacing (x,y,z)
  integer, dimension (3) :: N                       ! dimension in x,y,z
  real, dimension (:, :, :), allocatable :: f       ! array storing values of othe observable
  real fUnder, fOver
  contains
    procedure :: initgrid
    procedure :: copySetup
    procedure :: echoSetup
    procedure :: output
    procedure :: writePPM
    procedure :: writeXSF  
    procedure :: writeTAB  
    procedure :: fromXSF
	procedure :: interpolate => grid3D_interpolate
end type grid3D

contains 

! ===== grid3D :: initgrid ========
 subroutine initgrid ( this )
 implicit none
  class (grid3D) :: this
  this%Rspan(:) = this%Rmax(:) - this%Rmin(:)
  this%N(1) = floor ( this%Rspan(1)/ this%step(1) ) + 1
  this%N(2) = floor ( this%Rspan(2)/ this%step(2) ) + 1
  this%N(3) = floor ( this%Rspan(3)/ this%step(3) ) + 1
  allocate (this%f( this%N(1),this%N(2),this%N(3) ))
 end subroutine initgrid

! ===== grid3D :: setByTemplate ========
subroutine copySetup ( this, from )
 implicit none
  class (grid3D) :: this
  class (grid3D) :: from
  this%Rmin(:) = from%Rmin(:)
  this%Rmax(:) = from%Rmax(:)
  this%step(:) = from%step(:)
  call this%initgrid()
end subroutine copySetup

! ===== grid3D :: echoSetup ========
subroutine echoSetup ( this )
 implicit none
  class (grid3D) :: this
   write (*,'(A,3f16.8)') " step:  ", this%step (:)
   write (*,'(A,3f16.8)') " Rmin:  ", this%Rmin (:)
   write (*,'(A,3f16.8)') " Rmax:  ", this%Rmax (:)
   write (*,'(A,3f16.8)') " Rspan: ", this%Rspan(:)
   write (*,'(A,3i10)') " N: ", this%N(:)
end subroutine echoSetup

! ===== grid3D :: writePPM ========
subroutine writePPM(this, fname, maxf)
 implicit none
    class (grid3D) :: this
    character (*) fname
    real      ::  maxf
    integer :: i, j, k
    integer :: c
    write (*,*) "write PPN: ",fname,maxf
    open (unit = 69, file = fname, status = 'unknown')
    write(69, '(A2)') 'P3'
    write(69, '(1i4,1i4)') this%N(1), (this%N(2)+1)*this%N(3)
    write(69, '(A)') '255'
    do k=1, this%N(3)  ! layer 
    do j=1,  this%N(2)
       do i=1,  this%N(1)
        if(this%f(i,j,k) .lt. 0) then
          c= floor( min( 255.999, -255*this%f(i,j,k)/maxf))
          write (69,*) 255,255-c,255-c
        else
          c= floor( min( 255.999, +255*this%f(i,j,k)/maxf))
          write (69,*) 255-c,255-c,255
        end if
        ! write (2222,'(4i20,2f40.20)')  k,j,i,c, this%f(i,j,k), (-255*this%f(i,j,k)/maxf)  
       end do !i
    end do !j
    do i=1,  this%N(1)
       write (69,*) 0,0,0
    end do !i
    end do !k
  close (69)
end subroutine writePPM

! ===== grid3D :: writeXSF ========
subroutine writeXSF(this, fname)
 implicit none
    class (grid3D) :: this
    character (*) fname 
    integer :: i, j, k
    write (*,*) "write XSF: ",fname
    open (unit = 69, file = fname, status = 'unknown')
 write(69,*) "CRYSTAL"
 write(69,*) "PRIMVEC"
 write(69,'(3f12.6)')  this%Rspan(1), 0.0, 0.0
 write(69,'(3f12.6)')  0.0, this%Rspan(2), 0.0
 write(69,'(3f12.6)')  0.0, 0.0, this%Rspan(3)
 write(69,*) "CONVVEC"
 write(69,'(3f12.6)')  this%Rspan(1), 0.0, 0.0
 write(69,'(3f12.6)')  0.0, this%Rspan(2), 0.0
 write(69,'(3f12.6)')  0.0, 0.0, this%Rspan(3)
 write(69,*) "PRIMCOORD"
 write(69,*) "1  1"
 write(69,*) "1    0.000000    0.000000    0.00000"
 write(69,*)
 write(69,*) "BEGIN_BLOCK_DATAGRID_3D"
 write(69,*) "density_3D"                    
 write(69,*) " BEGIN_DATAGRID_3D_DENSITY"
 write(69,'(3i10)') this%N(:)
 write(69,'(3f12.6)')  0.0, 0.0, 0.0
 write(69,'(3f12.6)')  this%Rspan(1), 0.0, 0.0
 write(69,'(3f12.6)')  0.0, this%Rspan(2), 0.0
 write(69,'(3f12.6)')  0.0, 0.0, this%Rspan(3)
 do k=1, this%N(3)  ! layer 
    do j=1,  this%N(2)
       do i=1,  this%N(1)
         write(69,*) this%f(i,j,k)
       end do !i
    end do !j
 end do !k
 write(69,*) "END_DATAGRID_3D"
 write(69,*) "END_BLOCK_DATAGRID_3D"   
 close (69)
end subroutine writeXSF

! ===== grid3D :: writeTable ========
subroutine writeTAB(this, fname)
 implicit none
    class (grid3D) :: this
    character (*) fname 
    integer :: i, j, k
    write (*,*) "write TABLE: ",fname
    open (unit = 69, file = fname, status = 'unknown')
 do k=1, this%N(3)  ! layer 
    do j=1,  this%N(2)
       do i=1,  this%N(1)
         write(69,'(2x,e16.8)',advance='no') this%f(i,j,k)
       end do !i
       write(69,*)
    end do !j
   do i=1,  this%N(1)
      write(69,'(2x,e16.8)',advance='no') 0.0  ! row on 0.0
   end do ! i
   write(69,*)
 end do !k
 close (69)
end subroutine writeTAB

! ===== grid3D :: output =====
subroutine output(this, fname, iform, scaleV)
 implicit none
 class (grid3D) :: this
 character (*) fname 
 integer       iform
 real          scaleV
    if (btest(iform,0)) then
      !write (*,*) " write out ",(fname//'.ppm')
  	  call this%writePPM( fname//'.ppm',scaleV)
    end if
    if (btest(iform,1)) then
      !write (*,*) " write out ",(fname//'.xsf')
      call this%writeXSF( fname//".xsf" )
    end if  
    if (btest(iform,2)) then
      !write (*,*) " write out ",(fname//'.tab')
      call this%writeTAB( fname//".tab" )
    end if  
end  subroutine output

! ===== grid3D :: fromXSF =====
subroutine fromXSF(this, fname)
 implicit none
 class (grid3D) :: this
 character (*) fname 
 integer :: i, j, k
 integer iline
 character (100) line 
 real,    dimension (3,4) :: lvs 
 open (unit = 69, file = fname, status = 'old')
 do iline = 1,1000 ! search for BEGIN_DATAGRID_3D_
 	read (69,*) line
 	if ( index ( line, "BEGIN_DATAGRID_3D_") .ne. 0 ) exit
 end do
 read(69,*) this%N(:)
 read(69,*) lvs(:,1)
 read(69,*) lvs(:,2)
 read(69,*) lvs(:,3)
 read(69,*) lvs(:,4)
 	this%Rmin  (:) = lvs(:,1)
 	this%Rspan (1) = lvs(1,2)
 	this%Rspan (2) = lvs(2,3)
 	this%Rspan (3) = lvs(3,4)
 	this%Rmax  (:) = this%Rmin(:)  + this%Rspan(:)
 	this%step  (:) = this%Rspan(:) / this%N(:)
 	write (*,'(A,3f12.6)')  " step:  ",this%step
 	write (*,'(A,3f12.6)')  " Rmin:  ",this%Rmin
 	write (*,'(A,3f12.6)')  " Rmax:  ",this%Rmax
 	write (*,'(A,3f12.6)')  " Rspan: ",this%Rspan
 allocate ( this%f( this%N(1),this%N(2),this%N(3) ) )
 do k=1, this%N(3)  ! layer 
    do j=1,  this%N(2)
       do i=1,  this%N(1)
         read(69,*) this%f(i,j,k)
       end do !i
    end do !j
 end do !k
 close (69)
end subroutine fromXSF

! ===== interpolate =====
function grid3D_interpolate(this, R ) result(ff)
 implicit none
  ! variables
  class (grid3D) :: this
  real, dimension (3), intent (in)   :: R
  real                               :: ff
  integer x0,y0,z0
  integer x1,y1,z1
  real dx,dy,dz
  real mx,my,mz
  ! function body
  x0 = int( (R(1)/this%step(1)) + fastFloorOffset ) - fastFloorOffset
  z0 = int( (R(2)/this%step(2)) + fastFloorOffset ) - fastFloorOffset
  z0 = int( (R(3)/this%step(3)) + fastFloorOffset ) - fastFloorOffset
  dx = R(1) - x0*this%step(1)
  dy = R(2) - y0*this%step(2)
  dz = R(3) - z0*this%step(3)
  x0 = modulo( x0   , this%N(1) )+1
  y0 = modulo( y0   , this%N(2) )+1
  z0 = modulo( z0   , this%N(3) )+1
  x1 = modulo( x0+1 , this%N(1) )+1
  y1 = modulo( y0+1 , this%N(2) )+1
  z1 = modulo( z0+1 , this%N(3) )+1
  mx=1-dx
  my=1-dy
  mz=1-dz
  ! write (*,'(6f16.5,9i5)')   R(:), this%step(:), x0,y0,z0,   x1,y1,z1,  this%N(:) 
  ff    =    mz*(my*(mx*this%f(x0,y0,z0)     &
                    +dx*this%f(x1,y0,z0))    &
                +dy*(mx*this%f(x0,y1,z0)     &
                    +dx*this%f(x1,y1,z0)))   &
            +dz*(my*(mx*this%f(x0,y0,z1)     &
                    +dx*this%f(x1,y0,z1))    &
                +dy*(mx*this%f(x0,y1,z1)     &
                    +dx*this%f(x1,y1,z1)))
end function grid3D_interpolate

end module T_grid3D
